/**
 * Booking Hotels Search - Independent JavaScript Module
 * Clean, standalone implementation for booking hotels search
 * @version 2.0
 * @namespace BookingHotelsSearch
 */

(function() {
    'use strict';

    /**
     * BookingHotelsSearch Class
     * Handles search form submission, AJAX requests, and result display
     */
    class BookingHotelsSearch {
        constructor(options = {}) {
            this.options = {
                formSelector: options.formSelector || 'form.bc_form',
                resultsContainerSelector: options.resultsContainerSelector || '.ajax-search-result',
                resultCountSelector: options.resultCountSelector || '.result-count',
                searchUrl: options.searchUrl || null,
                ...options
            };

            this.form = null;
            this.resultsContainer = null;
            this.resultCount = null;
            this.isSubmitting = false;

            this.init();
        }

        /**
         * Initialize the search functionality
         */
        init() {
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => this.setup());
            } else {
                this.setup();
            }
        }

        /**
         * Setup event listeners and elements
         */
        setup() {
            // Find form
            this.form = document.querySelector(this.options.formSelector);
            if (!this.form) {
                console.warn('BookingHotelsSearch: Form not found');
                return;
            }

            // Find containers
            this.resultsContainer = document.querySelector(this.options.resultsContainerSelector);
            this.resultCount = document.querySelector(this.options.resultCountSelector);

            // Get search URL from form action
            if (!this.options.searchUrl) {
                this.options.searchUrl = this.form.action || window.location.href;
            }

            // Attach form submit handler
            this.form.addEventListener('submit', (e) => this.handleFormSubmit(e));

            // Validate form on input changes
            this.attachValidationListeners();

            console.log('✅ BookingHotelsSearch initialized');
        }

        /**
         * Attach validation listeners to form inputs
         */
        attachValidationListeners() {
            const startDateInput = this.form.querySelector('[name="start_date"]');
            const endDateInput = this.form.querySelector('[name="end_date"]');
            const adultsInput = this.form.querySelector('[name="adults"]');
            const childrenInput = this.form.querySelector('[name="children"]');

            // Date validation
            if (startDateInput && endDateInput) {
                const today = new Date().toISOString().split('T')[0];
                if (startDateInput.type === 'date') {
                    startDateInput.min = today;
                }

                startDateInput.addEventListener('change', () => {
                    if (endDateInput.type === 'date') {
                        endDateInput.min = startDateInput.value || today;
                    }
                    if (endDateInput.value && endDateInput.value <= startDateInput.value) {
                        endDateInput.value = '';
                    }
                });
            }

            // Children ages validation
            if (childrenInput) {
                childrenInput.addEventListener('change', () => {
                    this.updateChildAgeInputs(parseInt(childrenInput.value) || 0);
                });
            }

            // Initialize child age inputs if children > 0
            if (childrenInput && parseInt(childrenInput.value) > 0) {
                this.updateChildAgeInputs(parseInt(childrenInput.value));
            }
        }

        /**
         * Handle form submission
         */
        handleFormSubmit(e) {
            e.preventDefault();

            if (this.isSubmitting) {
                return false;
            }

            // Validate form
            if (!this.validateForm()) {
                return false;
            }

            // Perform search
            this.performSearch();
            return false;
        }

        /**
         * Validate form data
         */
        validateForm() {
            const errors = [];

            // Check dates
            const startDate = this.form.querySelector('[name="start_date"]')?.value;
            const endDate = this.form.querySelector('[name="end_date"]')?.value;

            if (!startDate || !endDate) {
                errors.push(this.getMessage('dates_required', 'Please select check-in and check-out dates'));
            } else {
                const start = new Date(startDate);
                const end = new Date(endDate);
                const today = new Date();
                today.setHours(0, 0, 0, 0);

                if (start < today) {
                    errors.push(this.getMessage('start_date_past', 'Check-in date cannot be in the past'));
                }

                if (end <= start) {
                    errors.push(this.getMessage('end_date_after_start', 'Check-out date must be after check-in date'));
                }
            }

            // Check adults
            const adults = parseInt(this.form.querySelector('[name="adults"]')?.value || 0);
            if (!adults || adults < 1) {
                errors.push(this.getMessage('adults_required', 'Please specify number of adults (minimum 1)'));
            }

            // Check children ages
            const children = parseInt(this.form.querySelector('[name="children"]')?.value || 0);
            if (children > 0) {
                const childAgeInputs = this.form.querySelectorAll('[name="child_ages[]"]');
                const filledAges = Array.from(childAgeInputs).filter(input => input.value !== '').length;

                if (filledAges < children) {
                    errors.push(this.getMessage('child_ages_required', `Please specify age for all ${children} children`));
                }
            }

            // Display errors
            if (errors.length > 0) {
                this.showError(errors.join('<br>'));
                return false;
            }

            return true;
        }

        /**
         * Perform AJAX search
         */
        performSearch() {
            if (this.isSubmitting) {
                return;
            }

            this.isSubmitting = true;

            // Prepare form data
            const formData = new FormData(this.form);
            formData.append('_ajax', '1');

            // Convert FormData to URLSearchParams for GET request
            const params = new URLSearchParams();
            for (const [key, value] of formData.entries()) {
                if (value !== '') {
                    if (params.has(key)) {
                        // Handle array parameters (child_ages[])
                        const existing = params.get(key);
                        params.set(key, existing + ',' + value);
                    } else {
                        params.append(key, value);
                    }
                }
            }

            // Show loading state
            this.showLoading();

            // Build request URL
            const url = this.options.searchUrl + '?' + params.toString();

            // Make AJAX request
            fetch(url, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                },
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                this.isSubmitting = false;
                if (data.status) {
                    this.handleSearchSuccess(data);
                } else {
                    this.showError(data.message || this.getMessage('search_error', 'An error occurred while searching'));
                }
            })
            .catch(error => {
                this.isSubmitting = false;
                console.error('Search error:', error);
                this.showError(this.getMessage('search_error', 'An error occurred while searching. Please try again.'));
            });
        }

        /**
         * Handle successful search response
         */
        handleSearchSuccess(response) {
            // Update results container
            if (response.data && response.data.fragments) {
                Object.entries(response.data.fragments).forEach(([selector, html]) => {
                    const element = document.querySelector(selector);
                    if (element) {
                        element.innerHTML = html;
                    }
                });
            }

            // Update URL without reloading
            this.updateURL();

            // Scroll to results
            if (this.resultsContainer) {
                this.resultsContainer.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }

            // Re-initialize any components in the results (e.g., date pickers, filters)
            this.reinitializeComponents();
        }

        /**
         * Update browser URL without reloading page
         */
        updateURL() {
            const formData = new FormData(this.form);
            const params = new URLSearchParams();

            // Add all form fields to URL
            for (const [key, value] of formData.entries()) {
                if (value !== '' && key !== '_ajax') {
                    if (params.has(key)) {
                        const existing = params.get(key);
                        params.set(key, existing + ',' + value);
                    } else {
                        params.append(key, value);
                    }
                }
            }

            // Update URL
            const newUrl = new URL(window.location.href);
            newUrl.search = params.toString();
            window.history.pushState({}, '', newUrl.toString());
        }

        /**
         * Show loading state
         */
        showLoading() {
            if (this.resultsContainer) {
                this.resultsContainer.innerHTML = `
                    <div class="text-center p-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p class="mt-3">${this.getMessage('searching', 'Searching hotels...')}</p>
                    </div>
                `;
            }

            // Disable submit button
            const submitBtn = this.form.querySelector('button[type="submit"], input[type="submit"]');
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.dataset.originalText = submitBtn.textContent || submitBtn.value;
                if (submitBtn.tagName === 'BUTTON') {
                    submitBtn.innerHTML = `<i class="fa fa-spinner fa-spin"></i> ${this.getMessage('searching', 'Searching...')}`;
                }
            }
        }

        /**
         * Show error message
         */
        showError(message) {
            if (this.resultsContainer) {
                this.resultsContainer.innerHTML = `
                    <div class="alert alert-danger" role="alert">
                        <strong>${this.getMessage('error', 'Error')}:</strong> ${message}
                    </div>
                `;
            } else {
                alert(message);
            }

            // Re-enable submit button
            const submitBtn = this.form.querySelector('button[type="submit"], input[type="submit"]');
            if (submitBtn) {
                submitBtn.disabled = false;
                if (submitBtn.tagName === 'BUTTON' && submitBtn.dataset.originalText) {
                    submitBtn.innerHTML = submitBtn.dataset.originalText;
                }
            }
        }

        /**
         * Update child age inputs based on children count
         */
        updateChildAgeInputs(childrenCount) {
            const container = this.form.querySelector('#child-ages-container');
            if (!container) {
                // Try to find or create container
                const childrenInput = this.form.querySelector('[name="children"]');
                if (childrenInput && childrenInput.closest('.form-group, .mb-3')) {
                    const parent = childrenInput.closest('.form-group, .mb-3');
                    let existingContainer = parent.querySelector('#child-ages-container');
                    if (!existingContainer) {
                        existingContainer = document.createElement('div');
                        existingContainer.id = 'child-ages-container';
                        existingContainer.className = 'child-ages-container mt-2';
                        parent.appendChild(existingContainer);
                    }
                    this.updateChildAgeInputsInContainer(existingContainer, childrenCount);
                }
                return;
            }

            this.updateChildAgeInputsInContainer(container, childrenCount);
        }

        /**
         * Update child age inputs in specific container
         */
        updateChildAgeInputsInContainer(container, childrenCount) {
            const existingInputs = container.querySelectorAll('.child-age-input');
            const currentCount = existingInputs.length;

            // Remove excess inputs
            if (currentCount > childrenCount) {
                for (let i = childrenCount; i < currentCount; i++) {
                    existingInputs[i].remove();
                }
            }

            // Add missing inputs
            if (currentCount < childrenCount) {
                for (let i = currentCount; i < childrenCount; i++) {
                    const ageInput = document.createElement('div');
                    ageInput.className = 'child-age-input mb-2';
                    ageInput.innerHTML = this.getChildAgeInputHTML(i + 1);
                    container.appendChild(ageInput);
                }
            }

            // Make all inputs required if children > 0
            container.querySelectorAll('[name="child_ages[]"]').forEach(input => {
                input.required = childrenCount > 0;
            });
        }

        /**
         * Get HTML for child age input
         */
        getChildAgeInputHTML(index) {
            const options = Array.from({length: 18}, (_, i) => {
                return `<option value="${i}">${i} ${this.getMessage('years', 'years')}</option>`;
            }).join('');

            return `
                <label class="form-label small">
                    ${index}. ${this.getMessage('child_age', 'Child Age')}:
                </label>
                <select name="child_ages[]" class="form-select form-select-sm" required>
                    <option value="">${this.getMessage('select', 'Select')}</option>
                    ${options}
                </select>
            `;
        }

        /**
         * Re-initialize components after AJAX update
         */
        reinitializeComponents() {
            // Re-initialize date validation
            this.attachValidationListeners();

            // Trigger any custom events for other scripts
            const event = new CustomEvent('bookingHotelsSearch:resultsUpdated', {
                detail: { container: this.resultsContainer }
            });
            document.dispatchEvent(event);
        }

        /**
         * Get translated message
         */
        getMessage(key, defaultValue) {
            // Check for global translations object
            if (window.translations && window.translations[key]) {
                return window.translations[key];
            }

            // Check for Laravel translations in window
            if (window.__ && typeof window.__ === 'function') {
                return window.__(key) || defaultValue;
            }

            return defaultValue;
        }
    }

    // Auto-initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            window.bookingHotelsSearch = new BookingHotelsSearch();
        });
    } else {
        window.bookingHotelsSearch = new BookingHotelsSearch();
    }

    // Export for manual initialization
    window.BookingHotelsSearch = BookingHotelsSearch;

})();
