/**
 * Booking Hotels - Main Module
 * Modern ES6+ JavaScript
 * @version 2.0
 */

class BookingHotelsApp {
    constructor() {
        this.state = {
            filters: {},
            loading: false,
            results: [],
        };

        this.init();
    }

    init() {
        // Initialize when DOM is ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.setup());
        } else {
            this.setup();
        }
    }

    setup() {
        console.log('🚀 Booking Hotels App initialized');

        // Init price range slider
        this.initPriceRangeSlider();

        // Init filter toggles
        this.initFilterToggles();

        // Init guest counter
        this.initGuestCounter();

        // Init child age inputs
        this.initChildAgeInputs();

        // Init date validation
        this.initDateValidation();

        // Init filter submission
        this.initFilterSubmission();

        // ✅ NEW: Init search form AJAX submission
        this.initSearchFormAjax();

        // ✅ Hotel detail "Available Rooms" (Blade) form support
        this.initHotelRoomSearchForm();

        // ✅ Offer selection (Rooms & Offers table) support
        this.initOfferSelectionTable();

        // ✅ Enhanced date range picker (like hotel-rooms-enhanced) for room-search-form
        this.initRoomSearchDateRangePicker();

        // ✅ Legacy-style guests dropdown (no Vue) for room-search-form
        this.initRoomSearchGuestsDropdown();

        // ✅ Room search (detail) AJAX: update URL + fragments
        this.initRoomSearchAjax();
    }

    initPriceRangeSlider() {
        const priceInput = document.querySelector('.filter-price');
        if (!priceInput || typeof $ === 'undefined' || typeof $.fn.ionRangeSlider === 'undefined') return;

        const min = parseInt(priceInput.dataset.min) || 0;
        const max = parseInt(priceInput.dataset.max) || 1000;
        const from = parseInt(priceInput.dataset.from) || min;
        const to = parseInt(priceInput.dataset.to) || max;
        const symbol = priceInput.dataset.symbol || '$';

        // Initialize Ion Range Slider
        $(priceInput).ionRangeSlider({
            type: 'double',
            min: min,
            max: max,
            from: from,
            to: to,
            prefix: symbol + ' ',
            prettify_enabled: true,
            prettify_separator: ',',
            onFinish: (data) => {
                priceInput.value = data.from + ';' + data.to;
            },
        });
    }

    initFilterToggles() {
        // Collapse/Expand filter items
        document.querySelectorAll('.g-filter-item .item-title').forEach(title => {
            title.addEventListener('click', (e) => {
                const item = e.currentTarget.closest('.g-filter-item');
                const content = item.querySelector('.item-content');

                if (content) {
                    const isVisible = content.style.display !== 'none';
                    content.style.display = isVisible ? 'none' : 'block';

                    const icon = title.querySelector('i');
                    if (icon) {
                        icon.classList.toggle('fa-angle-up');
                        icon.classList.toggle('fa-angle-down');
                    }
                }
            });
        });
    }

    initGuestCounter() {
        // Backward compatible (global) – used by main search
        this.setupCounter(document, '[name="adults"]', 1, 10);
        this.setupCounter(document, '[name="children"]', 0, 6, (value) => {
            this.updateChildAgeInputs(value);
        });
    }

    setupCounter(root, selector, min, max, onChange = null) {
        const input = (root || document).querySelector(selector);
        if (!input) return;

        const container = input.closest('.guest-counter');
        if (!container) return;

        const btnMinus = container.querySelector('.btn-minus');
        const btnPlus = container.querySelector('.btn-plus');

        if (!btnMinus || !btnPlus) return;

        btnMinus.addEventListener('click', (e) => {
            e.preventDefault();
            let value = parseInt(input.value) || min;
            if (value > min) {
                value--;
                input.value = value;
                if (onChange) onChange(value);
            }
        });

        btnPlus.addEventListener('click', (e) => {
            e.preventDefault();
            let value = parseInt(input.value) || min;
            if (value < max) {
                value++;
                input.value = value;
                if (onChange) onChange(value);
            }
        });
    }

    initChildAgeInputs() {
        const childrenInput = document.querySelector('[name="children"]');
        if (!childrenInput) return;

        const initialValue = parseInt(childrenInput.value) || 0;
        if (initialValue > 0) {
            this.updateChildAgeInputs(initialValue);
        }
    }

    updateChildAgeInputs(childrenCount) {
        const container = document.getElementById('child-ages-container');
        if (!container) return;

        const labelChildAge = container.dataset.i18nChildAge || 'Child Age';
        const labelSelect = container.dataset.i18nSelect || 'Select';
        const labelYears = container.dataset.i18nYears || 'years';

        // Clear existing inputs
        const existingInputs = container.querySelectorAll('.child-age-input');

        // If count decreased, remove excess inputs
        if (existingInputs.length > childrenCount) {
            for (let i = childrenCount; i < existingInputs.length; i++) {
                existingInputs[i].remove();
            }
        }

        // If count increased, add new inputs
        if (existingInputs.length < childrenCount) {
            for (let i = existingInputs.length; i < childrenCount; i++) {
                const ageInput = document.createElement('div');
                ageInput.className = 'child-age-input d-inline-block me-2 mb-2';
                ageInput.innerHTML = `
                    <label class="form-label small">
                        ${i + 1}. ${labelChildAge}:
                    </label>
                    <select name="child_ages[]" class="form-select form-select-sm" required>
                        <option value="">${labelSelect}</option>
                        ${Array.from({length: 18}, (_, idx) => idx).map(age =>
                            `<option value="${age}">${age} ${labelYears}</option>`
                        ).join('')}
                    </select>
                `;
                container.appendChild(ageInput);
            }
        }
    }

    initDateValidation() {
        const startDateInput = document.querySelector('[name="start_date"]');
        const endDateInput = document.querySelector('[name="end_date"]');

        if (!startDateInput || !endDateInput) return;

        // Set min date to today
        const today = new Date().toISOString().split('T')[0];
        startDateInput.min = today;

        startDateInput.addEventListener('change', () => {
            endDateInput.min = startDateInput.value;
            if (endDateInput.value && endDateInput.value <= startDateInput.value) {
                endDateInput.value = '';
            }
        });
    }

    /**
     * Hotel detail page - Available Rooms search (Blade)
     * The page has its own form `.room-search-form` which isn't `.bc_form`.
     * We attach counters + child ages + date validation scoped to that form.
     */
    initHotelRoomSearchForm() {
        const forms = document.querySelectorAll('form.room-search-form');
        if (!forms.length) return;

        forms.forEach((form) => {
            // In legacy-style form, inputs are hidden; the UI controls them.
            // We keep only child ages sync here (the dropdown handler calls updateChildAgeInputs).
            const hiddenChildren = form.querySelector('input[name="children"]');
            if (hiddenChildren) {
                const initialValue = parseInt(hiddenChildren.value) || 0;
                if (initialValue > 0) {
                    this.updateChildAgeInputs(initialValue);
                }
            }
        });
    }

    /**
     * Enhanced date range picker (daterangepicker) for booking-hotels detail page.
     * Mimics the "Check In - Out" UI from hotel-rooms-enhanced without Vue.
     */
    initRoomSearchDateRangePicker() {
        // Require jQuery + daterangepicker + moment (same stack as theme)
        if (typeof $ === 'undefined' || typeof $.fn.daterangepicker === 'undefined' || typeof moment === 'undefined') {
            return;
        }

        const form = document.querySelector('form.room-search-form');
        if (!form) return;

        const start = form.querySelector('input[name="start_date"]');
        const end = form.querySelector('input[name="end_date"]');
        const hidden = form.querySelector('input.bh-daterange-hidden');
        const render = form.querySelector('.bh-daterange-render');
        const text = form.querySelector('.bh-daterange-text');
        if (!start || !end || !hidden || !render) return;

        const setText = (s, e) => {
            const fmt = (window.bookingCore && window.bookingCore.date_format) ? window.bookingCore.date_format : 'YYYY-MM-DD';
            const txtVal = moment(s).format(fmt) + '  →  ' + moment(e).format(fmt);
            if (text) text.innerHTML = txtVal;
        };

        // Initial values
        const initialStart = start.value || hidden.value || moment().format('YYYY-MM-DD');
        const initialEnd = end.value || hidden.dataset.endDate || moment().add(1, 'day').format('YYYY-MM-DD');
        start.value = initialStart;
        end.value = initialEnd;
        setText(start.value, end.value);

        const options = {
            autoApply: true,
            showCalendar: false,
            opens: (window.bookingCore && window.bookingCore.rtl) ? 'left' : 'right',
            minDate: moment(),
            startDate: moment(start.value, 'YYYY-MM-DD'),
            endDate: moment(end.value, 'YYYY-MM-DD'),
            locale: {
                format: 'YYYY-MM-DD',
                direction: (window.bookingCore && window.bookingCore.rtl) ? 'rtl' : 'ltr',
            },
        };

        $(hidden).daterangepicker(options).on('apply.daterangepicker', (ev, picker) => {
            // Ensure at least 1 night
            if (picker.endDate.diff(picker.startDate, 'day') <= 0) {
                picker.endDate = picker.startDate.clone().add(1, 'day');
            }
            start.value = picker.startDate.format('YYYY-MM-DD');
            end.value = picker.endDate.format('YYYY-MM-DD');
            setText(start.value, end.value);
        });

        // Open picker when clicking render
        $(render).on('click', (e) => {
            e.preventDefault();
            $(hidden).trigger('click');
        });
    }

    /**
     * Guests dropdown (legacy-style, no Vue) for booking-hotels detail filter.
     * Keeps hidden inputs (adults/children) in sync and updates child ages UI.
     */
    initRoomSearchGuestsDropdown() {
        const form = document.querySelector('form.room-search-form');
        if (!form) return;

        const hiddenAdults = form.querySelector('input[name="adults"]');
        const hiddenChildren = form.querySelector('input[name="children"]');
        const adultsCountEl = form.querySelector('.bh-adults-count');
        const childrenCountEl = form.querySelector('.bh-children-count');
        const childrenAgesSection = document.getElementById('bh-child-ages-section');

        if (!hiddenAdults || !hiddenChildren) return;

        const clamp = (val, min, max) => Math.max(min, Math.min(max, val));
        const sync = () => {
            const a = clamp(parseInt(hiddenAdults.value || '1', 10) || 1, 1, 10);
            const c = clamp(parseInt(hiddenChildren.value || '0', 10) || 0, 0, 6);
            hiddenAdults.value = String(a);
            hiddenChildren.value = String(c);
            if (adultsCountEl) adultsCountEl.textContent = String(a);
            if (childrenCountEl) childrenCountEl.textContent = String(c);
            if (childrenAgesSection) {
                childrenAgesSection.style.display = c > 0 ? '' : 'none';
            }
            this.updateChildAgeInputs(c);
        };

        // Bind +/-
        form.querySelectorAll('.bh-plus, .bh-minus').forEach((btn) => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const isPlus = btn.classList.contains('bh-plus');
                const target = btn.getAttribute('data-input');
                if (target === 'adults') {
                    const next = (parseInt(hiddenAdults.value || '1', 10) || 1) + (isPlus ? 1 : -1);
                    hiddenAdults.value = String(next);
                }
                if (target === 'children') {
                    const next = (parseInt(hiddenChildren.value || '0', 10) || 0) + (isPlus ? 1 : -1);
                    hiddenChildren.value = String(next);
                }
                sync();
            });
        });

        // Bind manual inputs inside dropdown (optional)
        form.querySelectorAll('input.bh-input').forEach((inp) => {
            inp.addEventListener('change', () => {
                const target = inp.getAttribute('data-input');
                if (target === 'adults') hiddenAdults.value = inp.value;
                if (target === 'children') hiddenChildren.value = inp.value;
                sync();
            });
        });

        sync();
    }

    /**
     * AJAX search for hotel detail (no SPA):
     * - submit form.room-search-form with GET + _ajax=1
     * - replace #bh-room-search-filter and #bh-offers-container
     * - update URL via history.pushState (no reload)
     */
    initRoomSearchAjax() {
        if (typeof $ === 'undefined') return;

        const form = document.querySelector('form.room-search-form');
        if (!form) return;

        // Only for booking-hotels detail page offer-based mode
        const offersContainer = document.getElementById('bh-offers-container');
        const filterContainer = document.getElementById('bh-room-search-filter');
        if (!offersContainer || !filterContainer) return;

        $(form).off('submit.bh_roomsearch').on('submit.bh_roomsearch', (e) => {
            e.preventDefault();

            const $form = $(form);
            const url = $form.attr('action') || window.location.pathname;

            // Build query string (include child_ages[] selects)
            const formData = $form.serialize() + '&_ajax=1';

            // Loading state
            offersContainer.style.opacity = '0.6';

            $.ajax({
                url: url,
                type: 'GET',
                data: formData,
                success: (response) => {
                    if (response && response.status && response.data && response.data.fragments) {
                        const fragments = response.data.fragments;
                        Object.keys(fragments).forEach((selector) => {
                            const el = document.querySelector(selector);
                            if (el) el.innerHTML = fragments[selector];
                        });

                        // Update URL (remove _ajax)
                        const newUrl = new URL(window.location.href);
                        const params = new URLSearchParams(formData);
                        params.delete('_ajax');
                        newUrl.search = params.toString();
                        window.history.pushState({}, '', newUrl.toString());

                        // Re-init handlers after fragment replace
                        this.initHotelRoomSearchForm();
                        this.initRoomSearchDateRangePicker();
                        this.initRoomSearchGuestsDropdown();
                        this.initOfferSelectionTable();
                        this.initRoomSearchAjax();
                    } else {
                        // fallback: reload with params (without _ajax)
                        const newUrl = url + '?' + formData.replace('&_ajax=1', '');
                        window.location.href = newUrl;
                    }
                },
                error: () => {
                    const newUrl = url + '?' + formData.replace('&_ajax=1', '');
                    window.location.href = newUrl;
                },
                complete: () => {
                    offersContainer.style.opacity = '';
                },
            });
        });
    }

    /**
     * Rooms & Offers (Blade table) selection logic:
     * - validate at least one qty > 0 before submit
     * - compute total rooms + total price (like single-hotel.js computed total_price/total_rooms)
     * - show a small summary in card header and enable/disable submit button
     */
    initOfferSelectionTable() {
        const form = document.querySelector('form[action*="booking-hotels"][method="POST"]');
        if (!form) return;

        const qtySelects = form.querySelectorAll('select.bh-offer-qty');
        if (!qtySelects.length) return;

        const summary = document.getElementById('bh-offer-selection-summary');
        const submitBtn = form.querySelector('button[type="submit"], input[type="submit"]');

        const recompute = () => {
            let totalRooms = 0;
            let totalPrice = 0;
            let currency = '';
            const lines = [];

            qtySelects.forEach((sel) => {
                const qty = parseInt(sel.value || '0', 10) || 0;
                if (qty <= 0) return;

                const tr = sel.closest('tr');
                if (!tr) return;
                const lineTotal = parseFloat(tr.dataset.offerTotal || '0') || 0;
                currency = currency || (tr.dataset.offerCurrency || '');

                totalRooms += qty;
                totalPrice += (lineTotal * qty);

                const roomTitle = tr.dataset.roomTitle || '';
                const ratePlanName = tr.dataset.ratePlanName || '';
                const mealName = tr.dataset.mealName || '';
                const parts = [];
                if (roomTitle) parts.push(roomTitle);
                if (ratePlanName) parts.push(ratePlanName);
                if (mealName) parts.push(mealName);

                lines.push({
                    qty,
                    label: parts.join(' • ') || 'Selected offer',
                    subtotal: (lineTotal * qty),
                });
            });

            if (summary) {
                if (totalRooms > 0) {
                    summary.style.display = '';
                    const itemsHtml = lines.map((l) => {
                        const qtyLabel = `${l.qty}×`;
                        const price = `${l.subtotal.toFixed(2)} ${currency || ''}`.trim();
                        return `<li class="mb-1"><strong>${qtyLabel}</strong> ${this.escapeHtml(l.label)} <span class="text-muted">—</span> <strong>${price}</strong></li>`;
                    }).join('');

                    summary.innerHTML =
                        `<div class="d-flex flex-wrap align-items-center gap-2">` +
                        `<span class="badge bg-success">${totalRooms} ${totalRooms === 1 ? 'room' : 'rooms'}</span>` +
                        `<span><strong>Total:</strong> ${totalPrice.toFixed(2)} ${currency || ''}</span>` +
                        `</div>` +
                        `<div class="mt-2"><strong>Your selection:</strong></div>` +
                        `<ul class="mb-0 ps-3">${itemsHtml}</ul>`;
                } else {
                    summary.style.display = 'none';
                    summary.innerHTML = '';
                }
            }

            if (submitBtn) {
                submitBtn.disabled = totalRooms <= 0;
            }
        };

        // Initial state
        recompute();

        qtySelects.forEach((sel) => {
            sel.addEventListener('change', () => recompute());
        });

        form.addEventListener('submit', (e) => {
            // prevent submitting empty selection
            let any = false;
            qtySelects.forEach((sel) => {
                const qty = parseInt(sel.value || '0', 10) || 0;
                if (qty > 0) any = true;
            });
            if (!any) {
                e.preventDefault();
                alert('Please select at least one room.');
            }
        });
    }

    escapeHtml(str) {
        return String(str || '')
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    initFilterSubmission() {
        // Handle checkbox filters change
        document.querySelectorAll('.bc_filter input[type="checkbox"]').forEach(checkbox => {
            checkbox.addEventListener('change', this.debounce(() => {
                this.applyFilters();
            }, 300));
        });
    }

    applyFilters() {
        const form = document.querySelector('.bc_form_filter');
        if (!form) return;

        // Copy search params from main search form
        const searchForm = document.querySelector('.bc_form');
        if (searchForm) {
            ['location_id', 'start_date', 'end_date', 'adults', 'children'].forEach(name => {
                const searchInput = searchForm.querySelector(`[name="${name}"]`);
                if (searchInput && searchInput.value) {
                    let filterInput = form.querySelector(`[name="${name}"]`);
                    if (!filterInput) {
                        filterInput = document.createElement('input');
                        filterInput.type = 'hidden';
                        filterInput.name = name;
                        form.appendChild(filterInput);
                    }
                    filterInput.value = searchInput.value;
                }
            });

            // Copy child ages
            const existingChildAges = form.querySelectorAll('[name="child_ages[]"]');
            existingChildAges.forEach(input => input.remove());

            searchForm.querySelectorAll('[name="child_ages[]"]').forEach((input) => {
                if (input.value) {
                    const hiddenInput = document.createElement('input');
                    hiddenInput.type = 'hidden';
                    hiddenInput.name = 'child_ages[]';
                    hiddenInput.value = input.value;
                    form.appendChild(hiddenInput);
                }
            });
        }

        // Submit form
        form.submit();
    }

    initSearchFormAjax() {
        // Handle main search form submission with AJAX
        const searchForm = document.querySelector('form.bc_form');
        if (!searchForm) return;

        // Check if jQuery is available
        if (typeof $ === 'undefined') {
            console.warn('jQuery is required for AJAX search');
            return;
        }

        $(searchForm).on('submit', (e) => {
            e.preventDefault();

            const form = $(e.target);
            const url = form.attr('action');
            const formData = form.serialize() + '&_ajax=1';
            const resultsContainer = $('.ajax-search-result');
            const resultCount = $('.result-count');

            // Show loading
            if (resultsContainer.length) {
                resultsContainer.html(
                    '<div class="text-center p-5">' +
                    '<i class="fa fa-spinner fa-spin fa-3x"></i>' +
                    '<p>' + (window.translations?.searching || 'Searching...') + '</p>' +
                    '</div>'
                );
            }

            // Disable submit button
            const submitBtn = form.find('button[type="submit"], input[type="submit"]');
            const originalBtnText = submitBtn.html() || submitBtn.val();
            submitBtn.prop('disabled', true);
            if (submitBtn.html) {
                submitBtn.html('<i class="fa fa-spinner fa-spin"></i> ' + (window.translations?.searching || 'Searching...'));
            }

            $.ajax({
                url: url,
                type: 'GET',
                data: formData,
                success: (response) => {
                    if (response.status) {
                        // Update results
                        if (response.data && response.data.fragments) {
                            $.each(response.data.fragments, (selector, html) => {
                                $(selector).html(html);
                            });
                        }

                        // Update URL without reloading page
                        const newUrl = new URL(window.location.href);
                        const params = new URLSearchParams(formData);
                        params.delete('_ajax');
                        newUrl.search = params.toString();
                        window.history.pushState({}, '', newUrl.toString());
                    } else {
                        // Show error
                        if (resultsContainer.length) {
                            resultsContainer.html(
                                '<div class="alert alert-danger">' +
                                (response.message || (window.translations?.search_error || 'An error occurred while searching')) +
                                '</div>'
                            );
                        }
                    }
                },
                error: (xhr) => {
                    console.error('Search error:', xhr);
                    if (resultsContainer.length) {
                        resultsContainer.html(
                            '<div class="alert alert-danger">' +
                            (window.translations?.search_error || 'An error occurred while searching') +
                            '</div>'
                        );
                    }
                },
                complete: () => {
                    // Re-enable submit button
                    submitBtn.prop('disabled', false);
                    if (submitBtn.html) {
                        submitBtn.html(originalBtnText);
                    }
                }
            });
        });
    }

    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
}

// Initialize app
const bookingHotelsApp = new BookingHotelsApp();
