/**
 * Rooms Search Manager
 * Handles AJAX search for hotel rooms with rate plans
 *
 * Features:
 * - AJAX form submission
 * - Loading states
 * - Error handling
 * - URL updates
 * - Smooth animations
 */
class RoomsSearchManager {
    constructor() {
        this.form = null;
        this.isSubmitting = false;
        this.init();
    }

    init() {
        // Wait for DOM
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.setup());
        } else {
            this.setup();
        }
    }

    setup() {
        // Find form
        this.form = document.getElementById('booking-hotels-room-search-form');
        if (!this.form) {
            console.warn('Rooms search form not found');
            return;
        }

        // Check jQuery availability
        if (typeof $ === 'undefined') {
            console.error('jQuery is required for rooms search');
            return;
        }

        // Initialize date range picker if exists
        this.initDateRangePicker();

        // Initialize guests dropdown if exists
        this.initGuestsDropdown();

        // Handle form submission
        $(this.form).on('submit', (e) => this.handleSubmit(e));

        // ✅ NEW: Auto-trigger search if URL has params AND rooms container is empty
        this.autoTriggerSearchIfNeeded();
    }

    /**
     * ✅ NEW: Auto-trigger search if URL has search parameters but no rooms are displayed
     */
    autoTriggerSearchIfNeeded() {
        // Check if we have search parameters in URL
        if (!this.hasSearchParamsInURL()) {
            console.log('No search params in URL, skipping auto-trigger');
            return;
        }

        // Check if rooms container is empty
        if (!this.isRoomsContainerEmpty()) {
            console.log('Rooms already loaded, skipping auto-trigger');
            return;
        }

        console.log('Auto-triggering search from URL parameters...');
        
        // Small delay to ensure DOM is fully ready
        setTimeout(() => {
            this.handleSubmit({ preventDefault: () => {} });
        }, 500);
    }

    /**
     * ✅ NEW: Check if URL has search parameters
     */
    hasSearchParamsInURL() {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.has('start_date') || 
               urlParams.has('end_date') || 
               urlParams.has('adults') ||
               urlParams.has('suggested_room_id');
    }

    /**
     * ✅ NEW: Check if rooms container is empty (no room cards displayed)
     */
    isRoomsContainerEmpty() {
        const $container = $('#rooms-container');
        if (!$container.length) {
            return true; // Container not found = empty
        }
        
        // Check for room cards
        const $roomCards = $container.find('.room-with-plans-card, .room-card-highlighted');
        const isEmpty = $roomCards.length === 0;
        
        console.log('Rooms container check:', {
            containerExists: $container.length > 0,
            roomCardsCount: $roomCards.length,
            isEmpty: isEmpty
        });
        
        return isEmpty;
    }

        console.log('✅ Rooms Search Manager initialized');
    }

    /**
     * Initialize date range picker
     * Uses the same logic from main.js initRoomSearchDateRangePicker
     */
    initDateRangePicker() {
        // Require jQuery + daterangepicker + moment
        if (typeof $ === 'undefined' || typeof $.fn.daterangepicker === 'undefined' || typeof moment === 'undefined') {
            console.warn('daterangepicker or moment is not loaded');
            return;
        }

        const form = this.form;
        if (!form) return;

        const start = form.querySelector('input[name="start_date"]');
        const end = form.querySelector('input[name="end_date"]');
        const hidden = form.querySelector('input.bh-daterange-hidden');
        const render = form.querySelector('.bh-daterange-render');
        const text = form.querySelector('.bh-daterange-text');

        if (!start || !end || !hidden || !render) {
            console.warn('Date range picker elements not found');
            return;
        }

        const setText = (s, e) => {
            const fmt = (window.bookingCore && window.bookingCore.date_format) ? window.bookingCore.date_format : 'YYYY-MM-DD';
            const txtVal = moment(s).format(fmt) + '  →  ' + moment(e).format(fmt);
            if (text) text.innerHTML = txtVal;
        };

        // Initial values
        const initialStart = start.value || hidden.value || moment().format('YYYY-MM-DD');
        const initialEnd = end.value || hidden.dataset.endDate || moment().add(1, 'day').format('YYYY-MM-DD');
        start.value = initialStart;
        end.value = initialEnd;
        setText(start.value, end.value);

        const options = {
            autoApply: true,
            showCalendar: false,
            opens: (window.bookingCore && window.bookingCore.rtl) ? 'left' : 'right',
            minDate: moment(),
            startDate: moment(start.value, 'YYYY-MM-DD'),
            endDate: moment(end.value, 'YYYY-MM-DD'),
            locale: {
                format: 'YYYY-MM-DD',
                direction: (window.bookingCore && window.bookingCore.rtl) ? 'rtl' : 'ltr',
            },
        };

        $(hidden).daterangepicker(options).on('apply.daterangepicker', (ev, picker) => {
            // Ensure at least 1 night
            if (picker.endDate.diff(picker.startDate, 'day') <= 0) {
                picker.endDate = picker.startDate.clone().add(1, 'day');
            }
            start.value = picker.startDate.format('YYYY-MM-DD');
            end.value = picker.endDate.format('YYYY-MM-DD');
            setText(start.value, end.value);
        });

        // Open picker when clicking render
        $(render).on('click', (e) => {
            e.preventDefault();
            $(hidden).trigger('click');
        });
    }

    /**
     * Initialize guests dropdown
     * Uses the same logic from main.js initRoomSearchGuestsDropdown
     */
    initGuestsDropdown() {
        const form = this.form;
        if (!form) return;

        const hiddenAdults = form.querySelector('input[name="adults"]');
        const hiddenChildren = form.querySelector('input[name="children"]');
        const adultsCountEl = form.querySelector('.bh-adults-count');
        const childrenCountEl = form.querySelector('.bh-children-count');
        const childrenAgesSection = document.getElementById('bh-child-ages-section');

        if (!hiddenAdults || !hiddenChildren) return;

        const self = this;
        const clamp = (val, min, max) => Math.max(min, Math.min(max, val));
        const sync = () => {
            const a = clamp(parseInt(hiddenAdults.value || '1', 10) || 1, 1, 10);
            const c = clamp(parseInt(hiddenChildren.value || '0', 10) || 0, 0, 6);
            hiddenAdults.value = String(a);
            hiddenChildren.value = String(c);
            if (adultsCountEl) adultsCountEl.textContent = String(a);
            if (childrenCountEl) childrenCountEl.textContent = String(c);
            if (childrenAgesSection) {
                childrenAgesSection.style.display = c > 0 ? '' : 'none';
            }
            self.updateChildAgesInputs(c);
        };

        // Bind +/-
        form.querySelectorAll('.bh-plus, .bh-minus').forEach((btn) => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const isPlus = btn.classList.contains('bh-plus');
                const target = btn.getAttribute('data-input');
                if (target === 'adults') {
                    const next = (parseInt(hiddenAdults.value || '1', 10) || 1) + (isPlus ? 1 : -1);
                    hiddenAdults.value = String(clamp(next, 1, 10));
                }
                if (target === 'children') {
                    const next = (parseInt(hiddenChildren.value || '0', 10) || 0) + (isPlus ? 1 : -1);
                    hiddenChildren.value = String(clamp(next, 0, 6));
                }
                sync();
            });
        });

        // Bind manual inputs inside dropdown
        form.querySelectorAll('input.bh-input').forEach((inp) => {
            inp.addEventListener('change', () => {
                const target = inp.getAttribute('data-input');
                if (target === 'adults') {
                    hiddenAdults.value = String(clamp(parseInt(inp.value || '1', 10) || 1, 1, 10));
                }
                if (target === 'children') {
                    hiddenChildren.value = String(clamp(parseInt(inp.value || '0', 10) || 0, 0, 6));
                }
                sync();
            });
        });

        sync();
    }

    /**
     * Update child ages inputs based on children count
     */
    updateChildAgesInputs(childrenCount) {
        const $container = $('#child-ages-container');
        if (!$container.length) return;

        const currentInputs = $container.find('.child-age-input').length;
        const i18nChildAge = $container.data('i18n-child-age') || 'Child Age';
        const i18nSelect = $container.data('i18n-select') || 'Select';
        const i18nYears = $container.data('i18n-years') || 'years';

        if (childrenCount > currentInputs) {
            // Add new inputs
            for (let i = currentInputs; i < childrenCount; i++) {
                const $newInput = $(`
                    <div class="child-age-input d-inline-block me-2 mb-2">
                        <label class="form-label small">${i18nChildAge} ${i + 1}:</label>
                        <select name="child_ages[]" class="form-select form-select-sm" required>
                            <option value="">${i18nSelect}</option>
                            ${Array.from({ length: 18 }, (_, j) =>
                                `<option value="${j}">${j} ${i18nYears}</option>`
                            ).join('')}
                        </select>
                    </div>
                `);
                $container.append($newInput);
            }
        } else if (childrenCount < currentInputs) {
            // Remove extra inputs
            $container.find('.child-age-input').slice(childrenCount).remove();
        }
    }

    /**
     * Handle form submission
     */
    handleSubmit(e) {
        e.preventDefault();

        if (this.isSubmitting) {
            console.log('Search already in progress...');
            return false;
        }

        // Validate form
        if (!this.validateForm()) {
            return false;
        }

        // Get AJAX URL
        const ajaxUrl = $(this.form).data('ajax-url');
        if (!ajaxUrl) {
            // Fallback to regular form submission
            this.form.submit();
            return;
        }

        // Show loading
        this.showLoading();

        // Get form data
        const formData = $(this.form).serialize();

        // Update URL params
        this.updateUrlParams(formData);

        // Submit AJAX request
        this.isSubmitting = true;

        $.ajax({
            url: ajaxUrl,
            type: 'GET',
            data: formData,
            dataType: 'json',
            success: (response) => this.handleSuccess(response),
            error: (xhr) => this.handleError(xhr),
            complete: () => {
                this.isSubmitting = false;
                this.hideLoading();
            }
        });

        return false;
    }

    /**
     * Validate form
     */
    validateForm() {
        const startDate = $(this.form).find('[name="start_date"]').val();
        const endDate = $(this.form).find('[name="end_date"]').val();
        const adults = parseInt($(this.form).find('[name="adults"]').val()) || 0;
        const children = parseInt($(this.form).find('[name="children"]').val()) || 0;

        // Validate dates
        if (!startDate || !endDate) {
            alert(this.translate('Please select check-in and check-out dates'));
            return false;
        }

        const start = moment(startDate);
        const end = moment(endDate);

        if (!start.isValid() || !end.isValid()) {
            alert(this.translate('Please select valid dates'));
            return false;
        }

        if (end.isBefore(start) || end.isSame(start)) {
            alert(this.translate('Check-out date must be after check-in date'));
            return false;
        }

        // Validate adults
        if (adults < 1) {
            alert(this.translate('At least 1 adult is required'));
            return false;
        }

        // Validate child ages
        if (children > 0) {
            const childAges = $(this.form).find('[name="child_ages[]"]').map(function() {
                return $(this).val();
            }).get();

            const validAges = childAges.filter(age => age !== '' && age !== null);
            if (validAges.length !== children) {
                alert(this.translate('Please select age for all children'));
                return false;
            }
        }

        return true;
    }

    /**
     * Handle successful AJAX response
     */
    handleSuccess(response) {
        if (!response.status) {
            this.showError(response.message || this.translate('An error occurred while searching'));
            return;
        }

        // Update rooms container
        if (response.data && response.data.fragments) {
            Object.keys(response.data.fragments).forEach(selector => {
                const html = response.data.fragments[selector];
                const $target = $(selector);
                if ($target.length) {
                    $target.html(html);
                } else {
                    console.warn(`Selector ${selector} not found in DOM`);
                }
            });

            // Show rooms count alert
            const roomsCount = response.data.rooms_count || 0;
            const $countAlert = $('#rooms-count-alert');
            const $countText = $('#rooms-count-text');

            if (roomsCount > 0) {
                $countText.text(roomsCount);
                $countAlert.slideDown(300);
            } else {
                $countAlert.hide();
            }

            // Scroll to results
            const $container = $('#rooms-container');
            if ($container.length) {
                const offset = $container.offset();
                if (offset && offset.top) {
                    $('html, body').animate({
                        scrollTop: offset.top - 100
                    }, 500);
                }
            }

            // Re-initialize room selection manager if exists
            if (typeof window.roomSelectionManager !== 'undefined') {
                // Room selection manager should auto-update, but we can trigger update
                if (window.roomSelectionManager.updateSummary) {
                    window.roomSelectionManager.updateSummary();
                }
            }
        }
    }

    /**
     * Handle AJAX error
     */
    handleError(xhr) {
        let errorMessage = this.translate('An error occurred while searching. Please try again.');

        if (xhr.responseJSON && xhr.responseJSON.message) {
            errorMessage = xhr.responseJSON.message;
        } else if (xhr.status === 422 && xhr.responseJSON && xhr.responseJSON.errors) {
            // Validation errors
            const errors = xhr.responseJSON.errors;
            errorMessage = Array.isArray(errors) ? errors.join(', ') : errors;
        } else if (xhr.status === 404) {
            errorMessage = this.translate('Hotel not found');
        } else if (xhr.status === 500) {
            errorMessage = this.translate('Server error. Please try again later.');
        }

        this.showError(errorMessage);
    }

    /**
     * Show loading indicator
     */
    showLoading() {
        $('#rooms-loading-indicator').slideDown(300);
        $('#rooms-count-alert').hide();
        $('#rooms-container').fadeTo(200, 0.3);

        // Disable submit button
        const $submitBtn = $(this.form).find('button[type="submit"]');
        $submitBtn.prop('disabled', true);
        const originalText = $submitBtn.html();
        $submitBtn.data('original-text', originalText);
        $submitBtn.html(`<i class="fa fa-spinner fa-spin"></i> ${this.translate('Searching...')}`);
    }

    /**
     * Hide loading indicator
     */
    hideLoading() {
        $('#rooms-loading-indicator').slideUp(300);
        $('#rooms-container').fadeTo(200, 1);

        // Re-enable submit button
        const $submitBtn = $(this.form).find('button[type="submit"]');
        $submitBtn.prop('disabled', false);
        const originalText = $submitBtn.data('original-text');
        if (originalText) {
            $submitBtn.html(originalText);
        }
    }

    /**
     * Show error message
     */
    showError(message) {
        const $container = $('#rooms-container');
        const $errorAlert = $(`
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fa fa-exclamation-triangle"></i> ${this.escapeHtml(message)}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `);

        $container.prepend($errorAlert);

        // Auto-remove after 5 seconds
        setTimeout(() => {
            $errorAlert.fadeOut(() => $errorAlert.remove());
        }, 5000);
    }

    /**
     * Update URL parameters without page reload
     */
    updateUrlParams(formData) {
        const params = new URLSearchParams(formData);
        const newUrl = new URL(window.location.href);
        newUrl.search = params.toString();
        window.history.pushState({}, '', newUrl.toString());
    }

    /**
     * Translate string
     */
    translate(key) {
        const translations = {
            'Please select check-in and check-out dates': 'Please select check-in and check-out dates',
            'Please select valid dates': 'Please select valid dates',
            'Check-out date must be after check-in date': 'Check-out date must be after check-in date',
            'At least 1 adult is required': 'At least 1 adult is required',
            'Please select age for all children': 'Please select age for all children',
            'An error occurred while searching. Please try again.': 'An error occurred while searching. Please try again.',
            'Hotel not found': 'Hotel not found',
            'Server error. Please try again later.': 'Server error. Please try again later.',
            'Searching...': 'Searching...',
        };

        if (window.translations && window.translations[key]) {
            return window.translations[key];
        }

        return translations[key] || key;
    }

    /**
     * Escape HTML
     */
    escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return String(text || '').replace(/[&<>"']/g, m => map[m]);
    }
}

// Initialize when DOM is ready
if (typeof jQuery !== 'undefined') {
    $(document).ready(function() {
        if (typeof window.roomsSearchManager === 'undefined') {
            window.roomsSearchManager = new RoomsSearchManager();
        }
    });
} else {
    console.error('jQuery is required for RoomsSearchManager');
}
