/* global Vue */

(function () {
  function qsToObj(search) {
    const params = new URLSearchParams(search || window.location.search);
    const obj = {};
    for (const [k, v] of params.entries()) {
      obj[k] = v;
    }
    return obj;
  }

  function normalizeSearchParams(raw) {
    const p = { ...raw };

    if (p.start && !p.start_date) p.start_date = p.start;
    if (p.end && !p.end_date) p.end_date = p.end;

    p.adults = parseInt(p.adults || '2', 10);
    if (!Number.isFinite(p.adults) || p.adults < 1) p.adults = 2;

    p.children = parseInt(p.children || '0', 10);
    if (!Number.isFinite(p.children) || p.children < 0) p.children = 0;

    let ages = p.child_ages || '';
    if (typeof ages === 'string' && ages.trim() !== '') {
      ages = ages.split(',').filter(Boolean);
    }
    if (!Array.isArray(ages)) ages = [];
    ages = ages.map((x) => parseInt(x, 10)).filter((x) => Number.isFinite(x));
    while (ages.length < p.children) ages.push(5);
    if (ages.length > p.children) ages = ages.slice(0, p.children);
    p.child_ages = ages;

    return p;
  }

  function toQuery(params) {
    const q = new URLSearchParams();
    Object.entries(params).forEach(([k, v]) => {
      if (v === undefined || v === null || v === '') return;
      if (Array.isArray(v)) {
        v.forEach((item) => q.append(k + '[]', String(item)));
      } else {
        q.set(k, String(v));
      }
    });
    return q.toString();
  }

  async function apiGet(url, params) {
    const q = toQuery(params || {});
    const full = q ? `${url}?${q}` : url;
    const res = await fetch(full, {
      headers: { 'Accept': 'application/json' },
      credentials: 'same-origin',
    });
    const json = await res.json().catch(() => null);
    if (!res.ok) {
      const msg = json?.message || `HTTP ${res.status}`;
      throw new Error(msg);
    }
    return json;
  }

  const mount = document.getElementById('booking-hotels-spa-detail');
  if (!mount || typeof Vue === 'undefined') return;

  const slug = mount.dataset.slug;
  const initial = normalizeSearchParams(qsToObj(window.location.search));
  const initialToken = (new URLSearchParams(window.location.search)).get('offer_token') || '';

  Vue.createApp({
    data() {
      return {
        loading: false,
        error: null,
        hotel: null,
        search: initial,
        offerToken: initialToken,
        suggested: null,
        rooms: [],
      };
    },
    methods: {
      money(x) {
        const n = Number(x || 0);
        return n.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
      },
      checkoutUrl(token) {
        return `/booking-hotels/${encodeURIComponent(slug)}/checkout?offer_token=${encodeURIComponent(token)}`;
      },
      async loadAll() {
        this.loading = true;
        this.error = null;
        try {
          // Suggested (optional)
          if (this.offerToken) {
            const s = await apiGet(`/api/booking-hotels/${encodeURIComponent(slug)}/suggested-offer`, {
              offer_token: this.offerToken,
            });
            this.suggested = s?.data?.offer || null;
          }

          // Offers list
          const res = await apiGet(`/api/booking-hotels/${encodeURIComponent(slug)}/offers`, this.search);
          this.hotel = res?.data?.hotel || null;
          this.rooms = res?.data?.rooms || [];
        } catch (e) {
          this.error = e?.message || 'Failed to load';
          this.rooms = [];
        } finally {
          this.loading = false;
        }
      },
    },
    mounted() {
      this.loadAll();
    },
    template: `
      <div class="container" style="padding: 30px 0;">
        <div v-if="error" class="alert alert-danger">{{ error }}</div>
        <div v-if="loading" class="alert alert-secondary">Loading...</div>

        <div v-if="hotel" class="mb-4">
          <h2 class="mb-1">{{ hotel.title }}</h2>
          <div class="text-muted">{{ hotel.slug }}</div>
        </div>

        <div v-if="suggested" class="card mb-4 border-success">
          <div class="card-body">
            <div class="d-flex justify-content-between align-items-start">
              <div>
                <div class="text-success" style="font-weight:700;">Suggested offer</div>
                <div class="text-muted" style="font-size:13px;">
                  Room #{{ suggested.room_id }} • Rate plan #{{ suggested.rate_plan_id }}
                </div>
              </div>
              <div class="text-end">
                <div style="font-size: 18px; font-weight: 700;">
                  {{ money(suggested.pricing.total) }} {{ suggested.pricing.currency }}
                </div>
                <div class="text-muted" style="font-size:12px;">{{ suggested.pricing.nights }} nights</div>
              </div>
            </div>
            <div class="mt-3 d-flex justify-content-end">
              <a class="btn btn-success" :href="checkoutUrl(offerToken)">Book this offer</a>
            </div>
          </div>
        </div>

        <div v-if="rooms.length === 0 && !loading" class="alert alert-warning">No offers found for your search.</div>

        <div v-for="r in rooms" :key="r.room.id" class="card mb-3">
          <div class="card-body">
            <div class="d-flex justify-content-between align-items-start">
              <div>
                <h5 class="mb-1">{{ r.room.title }}</h5>
                <div class="text-muted" style="font-size:13px;">
                  Adults: {{ r.room.adults }} • Children: {{ r.room.children }} • Beds: {{ r.room.beds }}
                </div>
              </div>
            </div>

            <div class="mt-3">
              <div v-for="o in r.offers" :key="o.offer.offer_id" class="d-flex justify-content-between align-items-center border-top pt-2 mt-2">
                <div>
                  <div style="font-weight:600;">Offer</div>
                  <div class="text-muted" style="font-size:12px;">Qty: {{ o.offer.availability.available_qty }}</div>
                </div>
                <div class="text-end">
                  <div style="font-weight:700;">{{ money(o.offer.pricing.total) }} {{ o.offer.pricing.currency }}</div>
                  <div class="text-muted" style="font-size:12px;">{{ o.offer.pricing.nights }} nights</div>
                </div>
                <div class="ms-3">
                  <a class="btn btn-primary" :href="checkoutUrl(o.offer_token)">Book</a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    `,
  }).mount('#booking-hotels-spa-detail');
})();
